import 'dart:convert';
import 'dart:io';
import 'package:connectivity_plus/connectivity_plus.dart';
import 'package:flutter/material.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:hive_flutter/hive_flutter.dart';
import 'package:http/http.dart';
import 'package:zcart_delivery/helper/constants.dart';
import 'package:zcart_delivery/network/api.dart';

const _noInternetMsg = 'You are not connected to Internet';
//const _errorMsg = 'Please try again later.';

Future<bool> _isNetworkAvailable() async {
  var connectivityResult = await Connectivity().checkConnectivity();
  return connectivityResult != ConnectivityResult.none;
}

String getAccessToken() {
  var box = Hive.box(hiveBox);
  return box.get(access, defaultValue: '');
}

Future<void> setAccessToken(String token) async {
  var box = Hive.box(hiveBox);
  await box.put(access, token);
}

bool isRequestSuccessful(int code) {
  return code >= 200 && code <= 206;
}

class NetworkHelper {
  /// Variables
  bool accessAllowed = false;

  static Future<Response?> getRequest(String endPoint,
      {bool bearerToken = false, Map<String, dynamic>? requestBody}) async {
    if (await _isNetworkAvailable()) {
      Map<String, String>? headers;
      Response? response;

      var accessToken = getAccessToken();

      debugPrint("Access Token: $accessToken");

      if (bearerToken) {
        headers = {
          HttpHeaders.acceptHeader: 'application/json; charset=utf-8',
          'Authorization': 'Bearer $accessToken',
        };
      }

      try {
        final uri = requestBody == null
            ? Uri.parse('${API.base}$endPoint')
            : Uri.https(API.base.substring(8).split("/").first, "api/$endPoint",
                requestBody);

        debugPrint('URL: $uri');
        debugPrint('QueryParams: ${uri.queryParameters}');

        if (bearerToken) {
          response = await get(uri, headers: headers);
        } else {
          response = await get(uri);
        }
      } catch (e) {
        debugPrint(e.toString());
      }

      return response;
    } else {
      throw _noInternetMsg;
    }
  }

  static Future<Response?> postRequest(String endPoint, Map requestBody,
      {bool bearerToken = false}) async {
    if (await _isNetworkAvailable()) {
      Response? response;

      debugPrint('URL: ${API.base}$endPoint');
      debugPrint('body: $requestBody');

      var accessToken = getAccessToken();

      var headers = {
        HttpHeaders.acceptHeader: 'application/json; charset=utf-8',
      };

      if (bearerToken) {
        var header = {
          'Authorization': 'Bearer $accessToken',
        };
        headers.addAll(header);
      }

      debugPrint("Headers: $headers");
      try {
        response = await post(Uri.parse('${API.base}$endPoint'),
            body: requestBody, headers: headers);
      } catch (e) {
        debugPrint(e.toString());
      }

      return response;
    } else {
      throw _noInternetMsg;
    }
  }

  static Future<Response?> putRequest(String endPoint, Map request,
      {bool bearerToken = true}) async {
    if (await _isNetworkAvailable()) {
      Response? response;
      debugPrint('URL: ${API.base}$endPoint');
      debugPrint('Request: $request');

      var accessToken = getAccessToken();

      var headers = {
        HttpHeaders.acceptHeader: 'application/json; charset=utf-8',
      };

      if (bearerToken) {
        var header = {"Authorization": "Bearer $accessToken"};
        headers.addAll(header);
      }

      debugPrint("Headers: $headers");

      try {
        response = await put(Uri.parse('${API.base}$endPoint'),
            body: request, headers: headers);
      } catch (e) {
        debugPrint(e.toString());
      }
      debugPrint('Response: ${response?.statusCode} ${response?.body}');
      return response;
    } else {
      throw _noInternetMsg;
    }
  }

  // patchRequest(String endPoint, Map request,
  //     {bool requireToken = false,
  //     bool bearerToken = false,
  //     bool isDigitToken = false}) {}

  static Future<Response?> deleteRequest(String endPoint,
      {bool bearerToken = true}) async {
    if (await _isNetworkAvailable()) {
      var accessToken = getAccessToken();

      var headers = {
        HttpHeaders.acceptHeader: 'application/json; charset=utf-8',
      };

      if (bearerToken) {
        var header = {"Authorization": "Bearer $accessToken"};
        headers.addAll(header);
      }

      debugPrint(headers.toString());
      Response response =
          await delete(Uri.parse('${API.base}$endPoint'), headers: headers);
      debugPrint('Response: ${response.statusCode} ${response.body}');

      return response;
    } else {
      throw _noInternetMsg;
    }
  }
}

dynamic handleResponse(Response response, {bool showToast = true}) {
  if (isRequestSuccessful(response.statusCode)) {
    if (response.body.isNotEmpty) {
      debugPrint(response.statusCode.toString());
      debugPrint(response.body);
      return jsonDecode(response.body);
    } else {
      return response.body;
    }
  } else {
    debugPrint("handleResponse (json): ${jsonDecode(response.body)}");
    if (jsonDecode(response.body)['errors'] != null) {
      Fluttertoast.showToast(
          msg: jsonDecode(response.body)['errors']
              [jsonDecode(response.body)['errors'].keys.first][0]);
    } else if (showToast) {
      Fluttertoast.showToast(
          msg: jsonDecode(response.body)['message'] ??
              jsonDecode(response.body)['error']);
    }

    if (response.statusCode == 401) {
      setAccessToken('');
    }

    return response.statusCode;
  }
}
